<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Card;
use App\Models\Booking;
use App\Models\Transaction;
use App\Models\BookingSetting;
use App\Models\DropOfPoint;
use App\Traits\BookingFare;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use App\Models\BookingCharge;
// use Illuminate\Support\Str;
use App\Events\NewBookingEvent;
use Illuminate\Support\Facades\Schema;

class BookingController extends Controller
{
    use BookingFare;
    
    public function scheduleRide(Request $request)
    {
        
        // dd(now());
        
        //Validate the Input request 
        $validator =  $request->validate([
            'pickup_location' => 'required|string',
            'pickup_lat'    => 'required',
            'pickup_long'    => 'required',
            'drop_locations' => 'required|array',
            'pickup_date'    => 'required',
            'pickup_time'    => 'required',
            'car_plate_no'    => 'required',
            'notes'    => 'nullable',
           ]);
           
        try{
            
            $distance = $this->calculateDistance($request->pickup_lat, $request->pickup_long, $request->drop_locations[0]['drop_lat'], $request->drop_locations[0]['drop_long']);
            
            $pickupDateTime = Carbon::parse($request->pickup_date . ' ' . $request->pickup_time);
            
            $currentDateTime = Carbon::now();

            // Check if the pickup time is more than 2 weeks from now
            if ($pickupDateTime->gt($currentDateTime->addWeeks(2))) {
                return response()->json([
                    'success' => false,
                    'status' => 400,
                    'message' => 'Pickup time cannot be more than 2 weeks from the current time.',
                ], 200);
            }
           
            // Add 3 hours to the pickup time
            $dropOffTime = $pickupDateTime->addHours(3);
            
            //get the unique booking id 
            $bookingId = $this->generateUniqueBookingId();
           
    
              //Save the Booking Details
            $booking = Booking::create([
                'booking_id' => $bookingId,
                'rider_id'   => auth()->user()->id,
                'pickup_location' => $request->pickup_location,
                'pickup_date' => $request->pickup_date,
                'pickup_lat'  => $request->pickup_lat,
                'pickup_long' => $request->pickup_long,
                'pickup_time' => $request->pickup_time,
                'dropof_time' => $dropOffTime,
                'car_plate_no' => $request->car_plate_no,
                'distance'      => $distance,
                'notes' => $request->notes ? $request->notes: null,
            ]);
            
            //Save Drop Points for Booking Detail
            if($booking){
                
                $dropLocations = [];
                
                foreach ($request->drop_locations as $dl) {
                    $dropLocations[] = [
                        'booking_id' => $booking->id,
                        'drop_location' => $dl['drop_location'],
                        'drop_lat' => $dl['drop_lat'],
                        'drop_long' => $dl['drop_long'],
                    ];
                }
                //Save Drop Points
                DropOfPoint::insert($dropLocations);
            }
            
            $data = Booking::with('drop_points')->where('id', $booking->id)->first()->toArray();
            
            //calculate fare for the booking ride 
            list($baseFare , $platformFee, $driverFee , $totalFee, $additionalStops, $additionalStopsFee ) =   $this->fareCalculation($data);
            
            // update booking values to 
            $booking->update([
                'total_fare' => $totalFee,
                'base_fare' => $baseFare,
                'platform_fee' => $platformFee,
                'driver_fee' => $driverFee,
                'additional_stop_count' => $additionalStops,
                'additional_stop_fee' => $additionalStopsFee,
                ]);
                
                
                //Booking charges for rider on schedule ride
             $chargesDetails = [
                     'booking_id' => $booking->id,
                     'user_id' =>    auth()->user()->id,
                     'user_type' =>  User::TYPE_RIDER,
                     'amount' =>     $totalFee,
                     'charge_type' => 'debit',
                     'reason'       => 'booking_fee',
                 ]; 
             
             BookingCharge::create($chargesDetails);
                
                
            
            $bookingData = Booking::with('drop_points')->where('id', $booking->id)->first();
            
            event(new NewBookingEvent($bookingData));
             
        return $this->jsonResponse(true, 200, 'Ride Scheduled Successfully.', $bookingData);
             
        }catch(Exception $e){
             return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, [$e->getMessage()]);
        }
        
    }
    
    public function delayRide(Request $request)
    {
        
        
        //Validate the Input request 
        $validator =  $request->validate([
            'booking_id' => 'required',
           ]);
           
        try{
           
          $user = auth()->user();
          $bookingId = $request->booking_id;
          
           $bookingDetails = Booking::with('drop_points')->where('booking_id',$bookingId)->first();
           
             if($bookingDetails){
                 //if rider 
                if($user->user_type == User::TYPE_RIDER){
                     
                          // Get the previous delay count
                         $previousDelayCount = $bookingDetails->rider_delay_count;
                        
                        // Increment the delay count (for example, assume a delay occurred)
                        $currentDelayCount = $previousDelayCount + 1; // Increment count based on new delay
                            
                        $delayTime = BookingSetting::pluck('rider_delay_time')->first();
                                
                                if(is_null($delayTime)){
                                    $delayTime = "30";
                                }
                                
                                
                        //check if driver time conflicts then cancel the ride and relist 
                                
                        // Update the delay count and total delay time (for example, assume 30 minutes delay time)
                        $delayTimeIncrement = (int)$delayTime; // in minutes
                        $bookingDetails->rider_delay_count = $currentDelayCount;
                        $bookingDetails->rider_delay_time += $delayTimeIncrement;
                        
                        // Call the delayFare method to recalculate fare based on current delay increment
                        $this->riderDelayFare($bookingDetails, $currentDelayCount - $previousDelayCount);
                        
                        // // Save booking details after updating
                        // $bookingDetails->save();
                           $bookingDetails->is_delayed  = 1;
                           
                              if($bookingDetails->save()){
                                    return $this->jsonResponse(true, 200, 'Ride Delayed Successfully.');
                              }
                    
                 }   
                 //if driver 
            //   if($user->user_type == 2){
            //             // Get the previous delay count
            //              $previousDelayCount = $bookingDetails->driver_delay_count;
                        
            //             // Increment the delay count (for example, assume a delay occurred)
            //             $currentDelayCount = $previousDelayCount + 1; // Increment count based on new delay
                            
            //             $delayTime = BookingSetting::pluck('driver_delay_time')->first();
                                
            //                     if(is_null($delayTime)){
            //                         $delayTime = "30";
            //                     }
                                
            //             // Update the delay count and total delay time (for example, assume 30 minutes delay time)
            //             $delayTimeIncrement = (int)$delayTime; // in minutes
            //             $bookingDetails->driver_delay_count = $currentDelayCount;
            //             $bookingDetails->driver_delay_time += $delayTimeIncrement;
                        
            //             // Call the delayFare method to recalculate fare based on current delay increment
            //             $this->driverDelayFare($bookingDetails, $currentDelayCount - $previousDelayCount);
                        
            //             // // Save booking details after updating
            //             // $bookingDetails->save();
            //               $bookingDetails->is_delayed  = 1;
                           
            //                   if($bookingDetails->save()){
            //                     return $this->jsonResponse(true, 200, 'Ride Delayed Successfully.');
            //                   }
            //     }
             
             }else{
                return $this->jsonResponse(true, 200, 'No Record Found.');
             }
          
          
         
          
         
        }catch(Exception $e){
           return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, ["message" => $e->getMessage()]);
        }    
                   
        
    }
    
    public function acceptRide(Request $request)
    {
            
        $validator =  $request->validate([
            'booking_id' => 'required',
           ]);
       
        try{    
                $bookingId = $request->booking_id;
                
                $user = auth()->user();
                
                //check user type driver before accept the ride  
                if($user->user_type = User::TYPE_DRIVER){
                    
                    //Search the booking id in bookings and check if booking is pending to accept then show record
                   $booking = Booking::where('booking_id', $bookingId)->where('status',Booking::STATUS_PENDING)->first();

                    // Check if the booking exists and if driver_id is null
                    if ($booking && is_null($booking->driver_id)) {
                        
                      
                        
                        //check for the driver if he already accepted the ride for same day.
                        $isScheduledBooking = Booking::where('driver_id', $user->id)->where('pickup_date', date('Y-m-d'))->where('status',2)->first();
                        if($isScheduledBooking){
                            return $this->jsonResponse(true, 201, 'Booking already scheduled for the day.');
                        }
                        
                        // Assign the driver and update the driver_id
                        $booking->driver_id = $user->id;
                        $booking->status = 2;
                        $booking->save();
                        
                         return $this->jsonResponse(true, 200, 'Driver assigned successfully.');
                    } else {
                        return $this->jsonResponse(true, 400, 'Booking not found or driver already assigned.');
                    }
                    
                }else{
                    return $this->jsonResponse(false, 400, 'Role Doesn\'t Matches');
                }
                
        }catch(Exception $e){
             return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, ["message" => $e->getMessage()]);
        }  
        
    }
    
    
    public function changeBookingStatus(Request $request)
    {
            
        $validator =  $request->validate([
            'status' => 'required',
            'booking_id' => 'required'
           ]);
           
        try{ 
            
            $booking = Booking::where('booking_id', $request->booking_id)
            ->when(auth()->user()->type == 'driver', function ($query) {
                $query->where('driver_id', auth()->user()->id);
            })
            ->when(auth()->user()->type == 'rider', function ($query) {
                $query->where('rider_id', auth()->user()->id);
            })->first();
            
            
            if($booking){
                
                $fares = $this->fares();
                
                $status = $request->status;
                
                //if driver reached   
                if($status == 3){
                   $pickupDate = $booking->pickup_date;
                   $pickupTime = $booking->pickup_time;
                   
                       //get merged time from date and time 
                    $scheduledTime = Carbon::parse("$pickupDate $pickupTime");
                
                    $reachedAt = Carbon::now();
                    $reachedAtInTime = $reachedAt->format('H:i:s');
                   
                   
                    // Calculate the difference in minutes
                    $diffrenceTimeInMinutes = $reachedAt->diffInMinutes($scheduledTime);
                    //check if driver is late more than relaxation time 
                    
                    $driverLateGraceTime = BookingSetting::pluck('driver_late_grace')->first();
                    $driverLatePeriod    = BookingSetting::pluck('driver_late_period')->first();
                    
                    if($diffrenceTimeInMinutes > $driverLateGraceTime){
                        
                        $lateTime = $diffrenceTimeInMinutes - $driverLateGraceTime;
                        $lateCount =  ceil($lateTime % $driverLatePeriod);
                        $lateCharges = $lateCount * $fares['driver_late_fee'];
                        
                        // dd($lateTime, $lateCount ,$lateCharges);
                        
                             //Booking charges for driver
                             $chargesDetails = [
                                     'booking_id' => $booking->id,
                                     'user_id' =>    auth()->user()->id,
                                     'user_type' =>  User::TYPE_DRIVER,
                                     'amount' =>     $lateCharges,
                                     'charge_type' => 'debit',
                                     'reason'       => 'late_fee',
                                 ]; 
             
                         BookingCharge::create($chargesDetails);
             
                         $booking->update([
                            'driver_late_count' => $lateCount,
                            'driver_late_time' => $lateTime,
                            'driver_late_fee' => $lateCharges,
                            'driver_reached' => $reachedAtInTime,
                            'status' => $status
                        ]);  
                    }else{
                         $booking->update([
                            'driver_reached' => $reachedAtInTime,
                            'status' => $status
                            ]);  
                    }
                     
                }
                
                //if trip starts 
                if($status == 4){
                
                $startsAt = Carbon::now();
                $timeNow  = $startsAt->toTimeString(); 
                
                $booking->update([
                            'trip_start' => $timeNow,
                            'status' => $status
                            ]);  
                    
                    //check if rider is late more than relaxation time
                }
                
                //if trip ends or Complete Ride
                 if($status == 5){
                     
                     //confirm on hold booking payment 
                     $riderId = $booking->rider_id;
                     
                     $riderCardId = Card::where('user_id',$riderId)->pluck('stripe_payment_method_id')->first();
                     
                     //get booking transaction record where transaction status is on hold
                     $riderOnHoldTransaction = Transaction::where('booking_id',$booking->id)->where('user_id',$riderId)->where('status', Transaction::STATUS_ON_HOLD)->first();
                     if(!empty($riderOnHoldTransaction)){
                          $data['paymentIntentId'] = $riderOnHoldTransaction->transaction_id;
                          $data['card_id'] = $riderCardId;
                        
                         $confirmCharge =  $this->Confirmcharge($data);
                         if($confirmCharge['success'] == 1){
                            Transaction::where('transaction_id',$confirmCharge['chargeConfirmID'])->update(['status' => Transaction::STATUS_SUCCEEDED]);    
                         }
                     }
                    
                   
                     
                     //check for the charges to driver if any then deduct and pay final amount to driver
                     
                    $booking->update([
                    'trip_ends' => now(),
                    'status' => $request->status
                    ]);
                }
                
                
                   //if rider reached 
                 if($status == 8){
                    // dd($status);
                   $pickupDate = $booking->pickup_date;
                   $pickupTime = $booking->pickup_time;
                   
                       //get merged time from date and time 
                    $scheduledTime = Carbon::parse("$pickupDate $pickupTime");
                
                    $startsAt = Carbon::now();
                    $timeNow  = $startsAt->toTimeString(); 
                    
                    // Calculate the difference in minutes
                    $diffrenceTimeInMinutes = $startsAt->diffInMinutes($scheduledTime);
                    
                    $riderLateGraceTime = BookingSetting::pluck('rider_late_grace')->first();
                    $riderLatePeriod = BookingSetting::pluck('rider_late_period')->first();
                    
                    if($diffrenceTimeInMinutes > $riderLateGraceTime){
                        
                        $lateTime    = $diffrenceTimeInMinutes - $riderLateGraceTime;
                        $lateCount   =  ceil($lateTime % $riderLatePeriod);
                        $lateCharges = $lateCount * $fares['rider_late_fee'];
                        
                        // dd($lateTime, $lateCount ,$lateCharges);
                        
                         //Booking charges for driver
                             $chargesDetails = [
                                     'booking_id' => $booking->id,
                                     'user_id' =>    auth()->user()->id,
                                     'user_type' =>  User::TYPE_RIDER,
                                     'amount' =>     $lateCharges,
                                     'charge_type' => 'debit',
                                     'reason'       => 'late_fee',
                                 ]; 
             
                         BookingCharge::create($chargesDetails);
                        
                         $booking->update([
                            'rider_late_time' => $lateTime, 
                            'rider_late_fee' => $lateCharges,
                            'rider_reached' => $timeNow,
                            'status' => $status
                        ]);
                        
                    }else{
                         $booking->update([
                            'rider_reached' => $timeNow,
                            'status' => $status
                            ]);  
                    }
                    
                }
                
                    
                return $this->jsonResponse(true, 200, 'Ride Status Changed successfully.');
            }else{
                return $this->jsonResponse(true, 400, 'Booking not found Or Assigned to Other Driver.');
            }
            
        }catch(Exception $e){
            return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, ["message" => $e->getMessage()]);
        }  
        
    }
    
    //cancel ride for rider and driver with calculating there refund ammount 
    public function cancelRide(Request $request)
    {
             
        $validator =  $request->validate([
            'booking_id' => 'required'
           ]);
           
            try{
                // dd(now());
               $fares = $this->fares();
               
               $user = auth()->user();
               
               if($user->user_type == User::TYPE_DRIVER){
                   
                   //excluding states where we can't cancel the ride  
                   $excluded = [Booking::STATUS_CANCELED, Booking::STATUS_REFUNDED, Booking::STATUS_TRIP_ENDS];
                   
                   //fetch booking from bookings and check assigned for the same driver 
                   $bookingDetail = Booking::where('driver_id',$user->id)->where('booking_id', $request->booking_id)->whereNotIn('status', $excluded)->first();
                   
                   if($bookingDetail){
                        
                        
                        $pickupDate = $bookingDetail->pickup_date;
                        $pickupTime = $bookingDetail->pickup_time;
                        
                        //get merged time from date and time 
                        $scheduledTime = Carbon::parse("$pickupDate $pickupTime");
                    
                        $canceledAt = Carbon::now();
                    
                        // Calculate the difference in minutes
                        $diffrenceTimeInMinutes = $canceledAt->diffInMinutes($scheduledTime);
                        // dd($diffrenceTimeInMinutes);
                        if($diffrenceTimeInMinutes < 90){
                            //partial refund 
                            
                            $cancelationFee = $fares['driver_cancelation_fee'];
                            
                            // $this->driverCancelFare($bookingDetail);
                            
                            $bookingDetail->canceled_by = User::TYPE_DRIVER;
                            $bookingDetail->canceled_at = $canceledAt;
                            $bookingDetail->driver_cancelation_fee = $cancelationFee;
                            $bookingDetail->status = Booking::STATUS_CANCELED;
                            $bookingDetail->save();
                            
                           
                        }else{
                            //only update the canceled time stamps and status
                            $bookingDetail->canceled_by = User::TYPE_DRIVER;
                            $bookingDetail->canceled_at = $canceledAt;
                            $bookingDetail->driver_cancelation_fee = 0;
                            $bookingDetail->status = Booking::STATUS_CANCELED;
                            $bookingDetail->save();
                            
                            // dd("no fee.");
                            //refund to customer back
                            
                            
                            
                        }
                        
                         $response = [
                                    'success' => true,
                                    'status' => 200,
                                    'message' => 'Booking Canceled Successfully.',
                                ];
                        // dd($sheduledTime);
                        
        
                        //calculate fare of calculation and update in the table
                   }else{
                        $response = [
                                    'success' => false,
                                    'status' => 400,
                                    'message' => 'Booking not found For the driver or already canceled.',
                                ];
                   }
                   
                   
                   
                   
               }else{
                   
                    //excluding states where we can't cancel the ride  
                   $excluded = [Booking::STATUS_CANCELED, Booking::STATUS_REFUNDED, Booking::STATUS_TRIP_ENDS];
                   
                      //fetch booking from bookings and check assigned for the same rider 
                   $bookingDetail = Booking::where('rider_id',$user->id)->where('booking_id', $request->booking_id)->whereNotIn('status', $excluded)->first();
                   
                   
                   if($bookingDetail){
                        
                        $pickupDate = $bookingDetail->pickup_date;
                        $pickupTime = $bookingDetail->pickup_time;
                        
                        //get merged time from date and time 
                        $scheduledTime = Carbon::parse("$pickupDate $pickupTime");
                    
                        $canceledAt = Carbon::now();
                    
                        // Calculate the difference in minutes
                        $diffrenceTimeInMinutes = $canceledAt->diffInMinutes($scheduledTime);
                        
                         if($diffrenceTimeInMinutes < 90){
                            
                            // $cancelationFee = $fares['driver_cancelation_fee'];
                            // dd($bookingDetail);
                            // $this->riderCancelFare($bookingDetail);
                        
                           
                            
                            
                            $bookingDetail->canceled_by = User::TYPE_RIDER;
                            $bookingDetail->canceled_at = $canceledAt;
                            $bookingDetail->rider_cancelation_fee = $cancelationFee;
                            $bookingDetail->status = Booking::STATUS_CANCELED;
                            $bookingDetail->save();
                           
                        }else{
                            //   dd(" customer no fee.");
                            //only update the canceled time stamps and status
                            $bookingDetail->canceled_by = User::TYPE_RIDER;
                            $bookingDetail->canceled_at = $canceledAt;
                            $bookingDetail->rider_cancelation_fee = 0;
                            $bookingDetail->status = Booking::STATUS_CANCELED;
                            $bookingDetail->save();
                            
                          
                            //refund to customer back
                            
                            
                            
                        }
                         $response = [
                                   'success' => true,
                                    'status' => 200,
                                    'message' => 'Booking Canceled Successfully.',
                                ];
                       
                   }
                //   write the logic for the rider
               }
               
            return response()->json($response);
            
            }catch(Exception $e){
                
                return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, [$e->getMessage()]);
            } 
            
        
    }
    
    //rider booking History
    public function bookingHistory(){
        
            try{
               $bookings = Booking::with('drop_points')->where('rider_id', auth()->user()->id)->orderBy('id','desc')->get();
               
               if($bookings){
                   
                 return $this->jsonResponse(true, 200, 'Booking Record Fetched Successfully.',$bookings);
            
               }else{
                    return $this->jsonResponse(false, 400, 'No Booking Record Available!');
               }
               
                return response()->json($response);
                
            }catch(Exception $e){
                
                return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, [$e->getMessage()]);
            }
        
    }
    
    //rider current bookings details 
    public function currentBookings()
    {
             try{
                 //user activity 
                  User::where('id',auth()->user()->id)->update(['user_last_active' => now()]);
                   
                   $currentBookings = Booking::with('drop_points','driver','rider')->where('rider_id', auth()->user()->id)
                   ->whereDate('pickup_date', date('Y-m-d'))->whereNot('status',5)->whereNot('status',6)->whereNot('status',7)->get();
                
                if($currentBookings->isNotEmpty()){
                    
                    return $this->jsonResponse(true, 200, 'Current Bookings Fetched Successfully.', $currentBookings);
               }else{
                    return $this->jsonResponse(true, 200, 'No Booking Record Available!', []);
               }
                   
              }catch(Exception $e){
                  
                return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, ['message' => $e->getMessage()]);
            }
       
    }
    
    //driver upcoming Booking Details 
    public function upComingBookings()
    {
            try{
                
                $status = [Booking::STATUS_ACCEPTED, Booking::STATUS_DELAYED];
                $driverStatus = [Booking::DRIVER_STATUS_ACCEPTED, Booking::DRIVER_STATUS_REACHED,Booking::DRIVER_STATUS_START_TRIP];
                //Get upcoming Booking for the auth driver user where he alredy accepted the ride and having in above states given in array
                $upComingBookings = Booking::with('drop_points','rider')
                ->where('driver_id',auth()->user()->id)
                ->whereDate('pickup_date', '>=', date('Y-m-d'))
                ->whereTime('pickup_time', '>=', date('H:i:s'))
                ->whereIn('driver_status',$driverStatus)->whereIn('status',$status)->get();
                 if($upComingBookings->isNotEmpty()){
                        return $this->jsonResponse(true, 200, 'Upcoming Bookings Fetched Successfully.', $upComingBookings);
                   }else{
                        return $this->jsonResponse(true, 200, 'No Booking Record Available!', []);
                   }
                   
             }catch(Exception $e){
                 
                return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, [$e->getMessage()]);
            }
        
    }
    
    //get completed bookings for the driver
    public function completedBookings()
    {
             try{
                 $user = auth()->user();
                 if($user->user_type == User::TYPE_DRIVER){
                    $completedListStatus = [Booking::STATUS_COMPLETED,Booking::STATUS_CANCELED,Booking::STATUS_REFUNDED];
                    // dd($completedListStatus);
                     //get completed Bookings with status 5 that is completed rides for the auth driver 
                     $completedBookings  = Booking::with('drop_points')->where('driver_id',auth()->user()->id)
                     ->whereIn('status',$completedListStatus)->orderBy('id','desc')->get();
                     if($completedBookings->isNotEmpty()){
                         
                         return $this->jsonResponse(true, 200, 'Completed Bookings Fetched Successfully.', $completedBookings);
                         
                   }else{
                        return $this->jsonResponse(true, 200, 'No Completed Booking Record Available!', []);
                   }
               
                 }else{
                    return $this->jsonResponse(true, 400, 'User Type Does Not Matched!');
                 }
                
             }catch(Exception $e){
                 
                return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, [$e->getMessage()]);
     
            }
            
    }
    
    //available Ride Requests for driver
    public function availableRideRequests()
    {
        try{
            $request =  request()->all();
            $user = auth()->user();
            // dd($request);
            
                //user activity status
              User::where('id',auth()->user()->id)->update(['user_last_active' => now()]);
          
            $excludedStatuses = [Booking::STATUS_COMPLETED, Booking::STATUS_REFUNDED, Booking::STATUS_CANCELED];
            $availableBookings = Booking::with('drop_points','rider')    
             ->where(function($query) {
            $query->where('pickup_date', '>', date('Y-m-d'))
                      ->orWhere(function($query) {
                          $query->where('pickup_date', '=', date('Y-m-d'))
                            ->where('pickup_time', '>=', date('H:i:s'));
                          });
            })
            ->where(function ($query) {
                $query->whereNull('driver_id')
                      ->orWhere('driver_id', auth()->user()->id);
            })
            ->whereNotIn('status', $excludedStatuses);
            
            //sort by 
            if (request()->has('sort_by')) {
                
                // if ($request['sort_by'] == 1) {
                //     // Driver's current address
                //     $driverAddressLat = $user->address_lat;
                //     $driverAddressLong = $user->address_long;
    
                //     // Sort bookings based on nearest drop-off point
                //     $availableBookings->with(['drop_points' => function($query) use ($driverAddressLat, $driverAddressLong) {
                //         $query->selectRaw('*, 
                //             (6371 * acos(
                //                 cos(radians(?)) * cos(radians(drop_points.drop_lat)) * 
                //                 cos(radians(drop_points.drop_long) - radians(?)) + 
                //                 sin(radians(?)) * sin(radians(drop_points.drop_lat))
                //             )) as distance', 
                //             [$driverAddressLat, $driverAddressLong, $driverAddressLat]
                //         )
                //         ->orderBy('distance', 'asc');
                //     }]);
                // }
                
                 if ($request["sort_by"] == 4) {
                    // $availableBookings->orderBy('pickup_time');
                    // $user->lat
                }
                
                
                if ($request["sort_by"] == 4) {
                    $availableBookings->orderBy('pickup_time');      
                }
            
                if ($request["sort_by"] == 3) { 
                    $availableBookings->whereDate('pickup_date', date('Y-m-d'));  
                }
            }
            
            // filters 
            if(request()->has('filter') && request()->has('filter') == true ){
                if(request()->has('pickup_date_from') && request()->has('pickup_date_to')){
                    $pickupDateFrom = $request['pickup_date_from'];
                    $pickupDateTo = $request['pickup_date_to'];
                    // dd($pickupDateFrom,$pickupDateTo);
                    $availableBookings->whereBetween('pickup_date', [$pickupDateFrom, $pickupDateTo]);
                }
                if(request()->has(' ')){
                    $pickupTime = $request['pickup_time'];
                    $availableBookings->where('pickup_time',$pickupTime);
                }
            }
            
            if(request()->has('sort_by_date') && request()->has('sort_by_date') == true ){
            
                $availableBookings->orderby('pickup_date','desc');
            }
            if(request()->has('sort_by_time') && request()->has('sort_by_time') == true ){
            
                $availableBookings->orderby('pickup_time','desc');
            }
            
            
           $availableBookings = $availableBookings->orderby('created_at','desc')->get();
            // dd($availableBookings);
            
            // dd($availableBookings);
            
            $upcoming_trip_count  = Booking::where('driver_id',auth()->user()->id)
                ->whereDate('pickup_date', '=', date('Y-m-d'))
                ->whereTime('pickup_time', '>=', date('H:i:s'))
                ->whereNot('status',Booking::STATUS_PENDING)->whereNot('status',Booking::STATUS_COMPLETED)->count();
            
             if($availableBookings->isNotEmpty()){
                 
                     $responseData = [
                                'available_bookings' => $availableBookings,
                                'upcoming_trip_count' => $upcoming_trip_count,
                        ];
                        return $this->jsonResponse(true, 200, 'Available Ride Requests Fetched Successfully.', $responseData);
                  }else{
                        return $this->jsonResponse(true, 200, 'No Ride Request Record Available!', []);
                   }
            
        }catch(Exception $e){
            return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, ["message" => $e->getMessage()]);
        }
        
    }
    
    public function filterBookings()
    {
       $request =  request()->all(); 
        $user = auth()->user();
        
         $excludedStatuses = [Booking::STATUS_COMPLETED, Booking::STATUS_REFUNDED, Booking::STATUS_CANCELED];
            $availableBookings = Booking::with('drop_points','rider')    
             ->where(function($query) {
            $query->where('pickup_date', '>', date('Y-m-d'))
                      ->orWhere(function($query) {
                          $query->where('pickup_date', '=', date('Y-m-d'))
                            ->where('pickup_time', '>=', date('H:i:s'));
                          });
            })
            ->where(function ($query) {
                $query->whereNull('driver_id')
                      ->orWhere('driver_id', auth()->user()->id);
            })
            ->whereNotIn('status', $excludedStatuses);
            
            
                  //sort by 
            if (request()->has('sort_by')) {
                
                //Drop off Closest to my address
                 if ($request['sort_by'] == 1 && !empty($request['address_lat']) && !empty($request['address_long'])) {
                     
                    //get lat longs from request
                   $driverAddressLat =  $request['address_lat'];
                   $driverAddressLong = $request['address_long'];
                     
                     
                    // Driver's current address
                    // $driverAddressLat = $user->address_lat;
                    // $driverAddressLong = $user->address_long;
    
                    // Sort bookings based on nearest drop-off point
                    $availableBookings->with(['drop_points' => function($query) use ($driverAddressLat, $driverAddressLong) {
                        $query->selectRaw('*, 
                            (6371 * acos(
                                cos(radians(?)) * cos(radians(drop_lat)) * 
                                cos(radians(drop_long) - radians(?)) + 
                                sin(radians(?)) * sin(radians(drop_lat))
                            )) as distance', 
                            [$driverAddressLat, $driverAddressLong, $driverAddressLat]
                        )
                        ->orderBy('distance', 'asc');
                    }]);
                }else{
                   return $this->jsonResponse(true, 200, 'Please select closest address.');
                }
                
                //Pickup closest to my current location
                if ($request["sort_by"] == 2) {
                    // $availableBookings->orderBy('pickup_time');      
                    
                      $currentAddressLat = $user->latitude;
                        $currentAddressLong = $user->longitude;
                        
                        // dd($currentAddressLat,$currentAddressLong);
            
                        $availableBookings->selectRaw('*, 
                            (6371 * acos(
                                cos(radians(?)) * cos(radians(pickup_lat)) * 
                                cos(radians(pickup_long) - radians(?)) + 
                                sin(radians(?)) * sin(radians(pickup_lat))
                            )) as distance', 
                            [$currentAddressLat, $currentAddressLong, $currentAddressLat]
                        )
                        ->orderBy('distance', 'asc');
                    
                }
                
                if ($request["sort_by"] == 4) {
                    $availableBookings->orderBy('pickup_time');      
                }
            
                if ($request["sort_by"] == 3) { 
                    // $availableBookings->whereDate('created_', date('Y-m-d'));
                    $availableBookings->orderBy('created_at');
                }
            }
            
            // filters 
            if(request()->has('filter') && request()->has('filter') == true ){
                if(request()->has('pickup_date_from') && request()->has('pickup_date_to')){
                    $pickupDateFrom = $request['pickup_date_from'];
                    $pickupDateTo = $request['pickup_date_to'];
                    // dd($pickupDateFrom,$pickupDateTo);
                    $availableBookings->whereBetween('pickup_date', [$pickupDateFrom, $pickupDateTo]);
                }
                if(request()->has(' ')){
                    $pickupTime = $request['pickup_time'];
                    $availableBookings->where('pickup_time',$pickupTime);
                }
            }
            
            $availableBookings = $availableBookings->orderby('created_at','desc')->get();
            if($availableBookings->isNotEmpty()){
                    return $this->jsonResponse(true, 200, 'Rides Filtered Successfully.', $availableBookings);
              }else{
                    return $this->jsonResponse(true, 200, 'No Ride Available With Filter!', []);
               }
    }
    
    //getSingleBookingDetail
    public function getRideDetails(Request $request)
    {
         $validator =  $request->validate([
            'booking_id' => 'required'
           ]);
    
        try{
            $bookingDetails = Booking::with('drop_points','driver','rider','transactions')->where('booking_id', $request->booking_id)->get();
            
            if($bookingDetails){
                  return $this->jsonResponse(true, 200, 'Ride Detail Fetched Successfully.', $bookingDetails);
                }else{
                    return $this->jsonResponse(true, 200, 'No Record Available!', []);
                }
        
            }catch(Exception $e){
                return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, ["message" => $e->getMessage()]);
            }
        
    }
    
    
    function calculateDistance($pickupLat, $pickupLong, $dropLat, $dropLong) {
        
    // Convert latitude and longitude from degrees to radians
        $pickupLat = deg2rad($pickupLat);
        $pickupLong = deg2rad($pickupLong);
        $dropLat = deg2rad($dropLat);
        $dropLong = deg2rad($dropLong);
    
        // Haversine formula
        $dLat = $dropLat - $pickupLat;
        $dLong = $dropLong - $pickupLong;
    
        $a = sin($dLat / 2) * sin($dLat / 2) +
             cos($pickupLat) * cos($dropLat) *
             sin($dLong / 2) * sin($dLong / 2);
    
        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
    
        // Radius of Earth in kilometers. Use 3956 for miles
        $earthRadius = 6371;
    
        // Calculate the distance
        $distance = $earthRadius * $c;
    
        return round($distance, 2);
    }
    
    
    
    public function getUniqueBookingId()
    {
        $uniqueBookingId = $this->generateUniqueBookingId();
        
        $data = ["unique_booking_id" => $uniqueBookingId];
        
        return $this->jsonResponse(true, 200, 'Unique booking id generated.', $data);
        
    }
    
    public function getFaresDetail()
    {
       
        $faresData = $this->fares();
        
        return $this->jsonResponse(true, 200, 'Booking Fee List.', $faresData);
        
    }
    public function deleteUnpaidBookings(Request $request)
    { 
            try{
                
                $user = auth()->user();
                
                $bookings = Booking::where('rider_id',$user->id)->where('is_paid',0)->get();
                
                if($bookings->isNotEmpty()){
                   $bookings->each(function ($booking) {
                        $booking->delete();
                    });
                    
                return $this->jsonResponse(true, 200, 'Unpaid bookings deleted successfuly.');
                    
                }
                
            }catch(Exception $e){
                return $this->jsonResponse(false, 500, 'An error occurred.', null, 500, ["message" => $e->getMessage()]);
            }
        
    }
    
    function generateUniqueBookingId() {
    do {
            
            // Generate 4 random uppercase letters from A-Z
            $alphabetPart = implode('', array_map(function() {
                return chr(rand(65, 90)); // ASCII values for A-Z
            }, range(1, 4)));
            
            // Generate 4 random digits
            $numericPart = mt_rand(1000, 9999);
    
            // Combine alphabet and numeric parts
            $bookingId = $alphabetPart . $numericPart;
    
            // Check if this ID already exists in the bookings table
            $existingBooking = Booking::where('booking_id', $bookingId)->first();
        } while ($existingBooking); // Loop until a unique ID is found
    
        return $bookingId;
    }
}
